function stdcell_gui
%% ##### STD-CELL GUI #### %%
%% ========= VALORI DEFAULT =========
P = struct( ...
'L',24, 'W',10, 'H',2, ...
'origin',[0 0 0], ...
'H_Ti_bottom',1, ...
'H_gold_bottom',2, ...
'L_vac',4, 'H_vac',3, ...
'L_Au_top',16, 'W_Au_top',8, 'H_Au_top',4, ...
'T_film',0.5 ...
);
Via_default_nm = 1.0; % Via thickness [nm] per SDE
Phases_default = 1; % Phases number (integer >=1) per SDE
%% ========= GUI =========
ui = uifigure('Name','STD-CELL Editor','Color','w','Position',[100 100 1180 620]);
gl = uigridlayout(ui,[2 2]);
gl.RowHeight = {'1x', 46};
gl.ColumnWidth = {360, '1x'};
% --- Pannello controlli
ctrl = uipanel(gl,'Title','Parametri','FontWeight','bold');
ctrl.Layout.Row = 1;
ctrl.Layout.Column = 1;
cg = uigridlayout(ctrl,[16 2]);
cg.RowHeight = repmat({28},1,16);
cg.ColumnWidth = {190, 150};
% Campi
r=1;
ef_L = addField(cg, r, 'L [nm]', P.L, [0 Inf]); r=r+1;
ef_W = addField(cg, r, 'W [nm]', P.W, [0 Inf]); r=r+1;
ef_H = addField(cg, r, 'H [nm]', P.H, [0 Inf]); r=r+1;
ef_Hti = addField(cg, r, 'H_{Ti,bot} [nm]', P.H_Ti_bottom, [0 Inf]); r=r+1;
ef_Hau = addField(cg, r, 'H_{Au,bot} [nm]', P.H_gold_bottom,[0 Inf]); r=r+1;
ef_Lv = addField(cg, r, 'L_{vac} [nm]', P.L_vac, [0 Inf]); r=r+1;
ef_Hv = addField(cg, r, 'H_{vac} [nm]', P.H_vac, [0 Inf]); r=r+1;
ef_Lat = addField(cg, r, 'L_{Au,top} [nm]', P.L_Au_top, [0 Inf]); r=r+1;
ef_Wat = addField(cg, r, 'W_{Au,top} [nm]', P.W_Au_top, [0 Inf]); r=r+1;
ef_Hat = addField(cg, r, 'H_{Au,top} [nm]', P.H_Au_top, [0 Inf]); r=r+1;
ef_Tf = addField(cg, r, 'T_{film,Ti} [nm]',P.T_film, [0 Inf]); r=r+1;
% Origin (per la preview)
h = uilabel(cg,'Text','origin x0 [nm]','HorizontalAlignment','right','FontWeight','bold');
h.Layout.Row=r; h.Layout.Column=1;
ef_x0 = uieditfield(cg,'numeric','Value',P.origin(1),'Limits',[-Inf Inf]);
ef_x0.Layout.Row=r; ef_x0.Layout.Column=2; r=r+1;
h = uilabel(cg,'Text','origin y0 [nm]','HorizontalAlignment','right','FontWeight','bold');
h.Layout.Row=r; h.Layout.Column=1;
ef_y0 = uieditfield(cg,'numeric','Value',P.origin(2),'Limits',[-Inf Inf]);
ef_y0.Layout.Row=r; ef_y0.Layout.Column=2; r=r+1;
h = uilabel(cg,'Text','origin z0 [nm]','HorizontalAlignment','right','FontWeight','bold');
h.Layout.Row=r; h.Layout.Column=1;
ef_z0 = uieditfield(cg,'numeric','Value',P.origin(3),'Limits',[-Inf Inf]);
ef_z0.Layout.Row=r; ef_z0.Layout.Column=2; r=r+1;
% Extra per export SDE
ef_Via = addField(cg, r, 'Via thickness [nm]', Via_default_nm, [0 Inf]);
r=r+1;
h = uilabel(cg,'Text','Phases number','HorizontalAlignment','right','FontWeight','bold');
h.Layout.Row=r; h.Layout.Column=1;
ef_Ph = uieditfield(cg,'numeric','Value',Phases_default,'Limits',[1 Inf], ...
'RoundFractionalValues','on','ValueDisplayFormat','%.0f');
ef_Ph.Layout.Row=r; ef_Ph.Layout.Column=2;
% --- Bottoni in basso
btnPanel = uipanel(gl,'BorderType','none');
btnPanel.Layout.Row=2;
btnPanel.Layout.Column=[1 2];
bgl = uigridlayout(btnPanel,[1 6]);
bgl.ColumnWidth = {'fit','fit','1x','fit','fit','fit'};
uibutton(bgl,'Text','Reset default','ButtonPushedFcn',@(~,~)resetDefaults());
uibutton(bgl,'Text','Refresh','BackgroundColor',[0.1 0.5 0.1], ...
'FontColor','w','ButtonPushedFcn',@(~,~)updateAndDraw());
uilabel(bgl,'Text',''); % spacer
% Export SDE (.txt)
uibutton(bgl,'Text','Export SDE (.txt)','BackgroundColor',[0.10 0.35 0.70], ...
'FontColor','w','ButtonPushedFcn',@(~,~)exportSDE());
% Export Sdevice.txt
uibutton(bgl,'Text','Export Sdevice.txt','BackgroundColor',[0.80 0.35 0.10], ...
'FontColor','w','ButtonPushedFcn',@(~,~)exportSDevice());
uibutton(bgl,'Text','Close','ButtonPushedFcn',@(~,~)close(ui));
% --- Area grafica
axPanel = uipanel(gl,'Title','Anteprima','FontWeight','bold');
axPanel.Layout.Row=1;
axPanel.Layout.Column=2;
ax = uiaxes('Parent',axPanel,'BackgroundColor','w');
ax.Units = 'normalized';
ax.Position = [0 0 1 1];
% Aggiornamento automatico
allEf = [ef_L,ef_W,ef_H,ef_Hti,ef_Hau,ef_Lv,ef_Hv,ef_Lat,ef_Wat,ef_Hat,ef_Tf,ef_x0,ef_y0,ef_z0,ef_Via,ef_Ph];
for k=1:numel(allEf)
    allEf(k).ValueChangedFcn = @(~,~)updateAndDraw();
end
% Primo disegno
updateAndDraw();
%% ===== funzioni annidate GUI =====
function ef = addField(parent, row, lbl, val, lim)
        hLbl = uilabel(parent,'Text',lbl,'HorizontalAlignment','right','FontWeight','bold');
        hLbl.Layout.Row = row;
        hLbl.Layout.Column = 1;
        ef = uieditfield(parent,'numeric','Value',val, ...
'Limits',lim, 'LowerLimitInclusive','on','UpperLimitInclusive','on', ...
'ValueDisplayFormat','%.6g');
        ef.Layout.Row = row;
        ef.Layout.Column = 2;
end
function resetDefaults()
        P = struct('L',24,'W',10,'H',2,'origin',[0 0 0], ...
'H_Ti_bottom',1,'H_gold_bottom',2, ...
'L_vac',4,'H_vac',3, ...
'L_Au_top',16,'W_Au_top',8,'H_Au_top',4, ...
'T_film',0.5);
        ef_L.Value=P.L; ef_W.Value=P.W; ef_H.Value=P.H;
        ef_Hti.Value=P.H_Ti_bottom; ef_Hau.Value=P.H_gold_bottom;
        ef_Lv.Value=P.L_vac; ef_Hv.Value=P.H_vac;
        ef_Lat.Value=P.L_Au_top; ef_Wat.Value=P.W_Au_top; ef_Hat.Value=P.H_Au_top;
        ef_Tf.Value=P.T_film;
        ef_x0.Value=P.origin(1); ef_y0.Value=P.origin(2); ef_z0.Value=P.origin(3);
        ef_Via.Value = Via_default_nm;
        ef_Ph.Value = Phases_default;
        updateAndDraw();
end
function updateAndDraw()
        P.L = ef_L.Value; P.W = ef_W.Value; P.H = ef_H.Value;
        P.H_Ti_bottom = ef_Hti.Value; P.H_gold_bottom = ef_Hau.Value;
        P.L_vac = ef_Lv.Value; P.H_vac = ef_Hv.Value;
        P.L_Au_top = ef_Lat.Value; P.W_Au_top = ef_Wat.Value; P.H_Au_top = ef_Hat.Value;
        P.T_film = ef_Tf.Value;
        P.origin = [ef_x0.Value, ef_y0.Value, ef_z0.Value];
        % Clamp L_Au_top to max(L - 2*T_film, 0)
        max_Lat = max(0, P.L - 2 * P.T_film);
        if P.L_Au_top > max_Lat
            P.L_Au_top = max_Lat;
            ef_Lat.Value = max_Lat;
        end
        drawStack(ax, P);
end
function exportSDE()
% Converte nm -> µm (SDE usa µm)
        nm2um = @(x) x*1e-3;
        BottomDielectricWidth = nm2um(ef_L.Value);
        TrenchMiddleWidth = nm2um(ef_Lv.Value);
        BottomDielectricThickness = nm2um(ef_H.Value);
        CellLength = nm2um(ef_W.Value);
        TrenchWallHeight = nm2um(ef_Hv.Value);
        AdhesionLayerThickness = nm2um(ef_Hti.Value);
        GoldLayerThickness = nm2um(ef_Hau.Value);
        ViaThickness = nm2um(max(ef_Via.Value,0));
        PhasesNumber = max(1, round(ef_Ph.Value));
        TopGoldWidthX = nm2um(ef_Lat.Value);
        TopGoldHeightY = nm2um(ef_Wat.Value);
        TopGoldThicknessZ = nm2um(ef_Hat.Value);
        TitaniumRimThickness = nm2um(ef_Tf.Value);
% Testo SDE esattamente come richiesto, con numeri aggiornati
        content = sprintf([ ...
';; ============================================================\n' ...
';; SENTARUS SDE - STANDARD CELL\n' ...
';; SiO2 — Dielettrico HfO2 — Cornice Ti — Contacts Au/Ti\n' ...
';; ============================================================\n\n' ...
'(sde:clear)\n' ...
';; -------------------------------\n' ...
';; MAIN PARAMETER\n' ...
';; -------------------------------\n' ...
'(sde:define-parameter "BottomDielectricWidth" %.6f)\n' ...
'(sde:define-parameter "TrenchMiddleWidth" %.6f)\n' ...
'(sde:define-parameter "TrenchWallWidth" (/ (- BottomDielectricWidth TrenchMiddleWidth) 2))\n' ...
'(sde:define-parameter "BottomDielectricThickness" %.6f)\n' ...
'(sde:define-parameter "CellLength" %.6f)\n' ...
'(sde:define-parameter "TrenchWallHeight" %.6f)\n' ...
'(sde:define-parameter "AdhesionLayerThickness" %.6f)\n' ...
'(sde:define-parameter "GoldLayerThickness" %.6f)\n' ...
'(sde:define-parameter "ViaThickness" %.6f)\n' ...
'(sde:define-parameter "PhasesNumber" %d)\n\n' ...
';; -------------------------------\n' ...
';; TOP GOLD + CORNER\n' ...
';; -------------------------------\n' ...
'(sde:define-parameter "TopGoldWidthX" %.6f)\n' ...
'(sde:define-parameter "TopGoldHeightY" %.6f)\n' ...
'(sde:define-parameter "TopGoldThicknessZ" %.6f)\n' ...
'(sde:define-parameter "TitaniumRimThickness" %.6f)\n\n' ...
';; -------------------------------\n' ...
';; SECONDARY PARAMETER\n' ...
';; -------------------------------\n' ...
'(sde:define-parameter "x_2ndWallStart" (+ TrenchWallWidth TrenchMiddleWidth))\n' ...
'(sde:define-parameter "z_BottomAdhesionLayerEnd" (+ BottomDielectricThickness AdhesionLayerThickness))\n' ...
'(sde:define-parameter "z_MiddleGoldEnd" (+ BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness))\n' ...
'(sde:define-parameter "z_WallEnd" (+ z_MiddleGoldEnd TrenchWallHeight))\n' ...
'(sde:define-parameter "z_2ndGoldLayerEnd" (+ z_WallEnd TopGoldThicknessZ))\n' ...
'(sde:define-parameter "CenterX" (/ BottomDielectricWidth 2.0))\n' ...
'(sde:define-parameter "CenterY" (/ CellLength 2.0))\n\n' ...
';; Bounding boxes\n' ...
'(sde:define-parameter "x_Au_min" (- CenterX (/ TopGoldWidthX 2.0)))\n' ...
'(sde:define-parameter "x_Au_max" (+ CenterX (/ TopGoldWidthX 2.0)))\n' ...
'(sde:define-parameter "y_Au_min" (- CenterY (/ TopGoldHeightY 2.0)))\n' ...
'(sde:define-parameter "y_Au_max" (+ CenterY (/ TopGoldHeightY 2.0)))\n' ...
'(sde:define-parameter "x_Rim_min" (- x_Au_min TitaniumRimThickness))\n' ...
'(sde:define-parameter "x_Rim_max" (+ x_Au_max TitaniumRimThickness))\n' ...
'(sde:define-parameter "y_Rim_min" (- y_Au_min TitaniumRimThickness))\n' ...
'(sde:define-parameter "y_Rim_max" (+ y_Au_max TitaniumRimThickness))\n\n' ...
';; ============================================================\n' ...
';; BOTTOM STRUCTURE\n' ...
';; ============================================================\n' ...
'(sdegeo:create-cuboid (position 0 0 0)\n' ...
' (position BottomDielectricWidth CellLength BottomDielectricThickness)\n' ...
' "SiO2")\n\n' ...
'(sdegeo:create-cuboid (position 0 0 BottomDielectricThickness)\n' ...
' (position BottomDielectricWidth CellLength z_BottomAdhesionLayerEnd)\n' ...
' "Titanium")\n\n' ...
'(define BOTTOMCONTACT\n' ...
' (sdegeo:create-cuboid (position 0 0 z_BottomAdhesionLayerEnd)\n' ...
' (position BottomDielectricWidth CellLength z_MiddleGoldEnd)\n' ...
' "Gold"))\n\n' ...
';; ============================================================\n' ...
';; CENTRAL STRUCTURE\n' ...
';; ============================================================\n' ...
'(sdegeo:create-cuboid (position TrenchWallWidth 0 z_MiddleGoldEnd)\n' ...
' (position x_2ndWallStart CellLength z_WallEnd)\n' ...
' "Vacuum")\n\n' ...
'(define BODY_LEFT\n' ...
' (sdegeo:create-cuboid (position 0 0 z_MiddleGoldEnd)\n' ...
' (position TrenchWallWidth CellLength z_WallEnd)\n' ...
' "HfO2"))\n\n' ...
'(define BODY_RIGHT\n' ...
' (sdegeo:create-cuboid (position x_2ndWallStart 0 z_MiddleGoldEnd)\n' ...
' (position BottomDielectricWidth CellLength z_WallEnd)\n' ...
' "HfO2"))\n\n' ...
';; ============================================================\n' ...
';; TOP STRUCTURE\n' ...
';; ============================================================\n' ...
'(define TOPCONTACT\n' ...
' (sdegeo:create-cuboid (position x_Au_min y_Au_min z_WallEnd)\n' ...
' (position x_Au_max y_Au_max z_2ndGoldLayerEnd)\n' ...
' "Gold"))\n\n' ...
';; Cornice Ti (4 lati)\n' ...
'(define RIM_LEFT (sdegeo:create-cuboid (position x_Rim_min y_Rim_min z_WallEnd)\n' ...
' (position x_Au_min y_Rim_max z_2ndGoldLayerEnd)\n' ...
' "Titanium"))\n' ...
'(define RIM_RIGHT (sdegeo:create-cuboid (position x_Au_max y_Rim_min z_WallEnd)\n' ...
' (position x_Rim_max y_Rim_max z_2ndGoldLayerEnd)\n' ...
' "Titanium"))\n' ...
'(define RIM_BOTTOM (sdegeo:create-cuboid (position x_Rim_min y_Rim_min z_WallEnd)\n' ...
' (position x_Rim_max y_Au_min z_2ndGoldLayerEnd)\n' ...
' "Titanium"))\n' ...
'(define RIM_TOP (sdegeo:create-cuboid (position x_Rim_min y_Au_max z_WallEnd)\n' ...
' (position x_Rim_max y_Rim_max z_2ndGoldLayerEnd)\n' ...
' "Titanium"))\n\n' ...
'(sdegeo:bool-unite (list RIM_LEFT RIM_RIGHT RIM_BOTTOM RIM_TOP))\n\n' ...
';; ============================================================\n' ...
';; TOP STRUCTURE - SiO2 AROUND Ti+Au\n' ...
';; ============================================================\n' ...
'(define FILL_BOTTOM (sdegeo:create-cuboid (position 0 0 z_WallEnd)\n' ...
' (position BottomDielectricWidth y_Rim_min z_2ndGoldLayerEnd)\n' ...
' "SiO2"))\n' ...
'(define FILL_TOP (sdegeo:create-cuboid (position 0 y_Rim_max z_WallEnd)\n' ...
' (position BottomDielectricWidth CellLength z_2ndGoldLayerEnd)\n' ...
' "SiO2"))\n' ...
'(define FILL_LEFT (sdegeo:create-cuboid (position 0 y_Rim_min z_WallEnd)\n' ...
' (position x_Rim_min y_Rim_max z_2ndGoldLayerEnd)\n' ...
' "SiO2"))\n' ...
'(define FILL_RIGHT (sdegeo:create-cuboid (position x_Rim_max y_Rim_min z_WallEnd)\n' ...
' (position BottomDielectricWidth y_Rim_max z_2ndGoldLayerEnd)\n' ...
' "SiO2"))\n' ...
'(sdegeo:bool-unite (list FILL_BOTTOM FILL_TOP FILL_LEFT FILL_RIGHT))\n\n' ...
'\n' ...
';; ============================================================\n' ...
';; CONTACTS\n' ...
';; ============================================================\n\n' ...
'(sdegeo:define-contact-set "BottomContact" 4 (color:rgb 1 0 0) "##")\n' ...
'(sdegeo:set-current-contact-set "BottomContact")\n' ...
'(sdegeo:set-contact BOTTOMCONTACT "BottomContact")\n\n' ...
'(sdegeo:define-contact-set "TopContact" 4 (color:rgb 0 0 1) "##")\n' ...
'(sdegeo:set-current-contact-set "TopContact")\n' ...
'(sdegeo:set-contact TOPCONTACT "TopContact");;\n\n' ...
'\n' ...
';; ============================================================\n' ...
';; MESH / REFINEMENT\n' ...
';; ============================================================\n' ...
'(sdedr:define-refeval-window "InnerTrenchMesh" "Cuboid"\n' ...
' (position TrenchWallWidth 0 z_BottomAdhesionLayerEnd)\n' ...
' (position x_2ndWallStart CellLength z_2ndGoldLayerEnd))\n\n' ...
'(sdedr:define-refinement-size "TrenchRefinement" 0.001 0.005 0.001 0.001 0.005 0.001)\n' ...
'(sdedr:define-refinement-placement "RefinementPlacement_1" "TrenchRefinement"\n' ...
' (list "window" "InnerTrenchMesh"))\n' ...
'(sdedr:define-refinement-function "TrenchRefinement" "MaxLenInt" "Gold" "Vacuum" 0.0001 "DoubleSide")\n' ...
'(sdedr:define-refinement-function "TrenchRefinement" "MaxLenInt" "Gold" "Titanium" 0.0001)\n\n' ...
';; ============================================================\n' ...
';; BUILD + SAVE STRUCTURE\n' ...
';; ============================================================\n' ...
'(sde:build-mesh "n@node@")\n' ...
        ], ...
        BottomDielectricWidth, TrenchMiddleWidth, BottomDielectricThickness, ...
        CellLength, TrenchWallHeight, AdhesionLayerThickness, ...
        GoldLayerThickness, ViaThickness, PhasesNumber, ...
        TopGoldWidthX, TopGoldHeightY, TopGoldThicknessZ, TitaniumRimThickness);
        [f,p] = uiputfile('stdcell_sde_input_command.txt','Export SDE .txt');
if isequal(f,0), return; end
        fid = fopen(fullfile(p,f),'w');
if fid<0
            uialert(ui,'Impossibile creare il file di output.','Errore','Icon','error');
return;
end
        fwrite(fid, content, 'char');
        fclose(fid);
        uialert(ui, sprintf('File generato:\n%s', fullfile(p,f)), 'Export completato', 'Icon', 'success');
end
function exportSDevice()
% Deck SDevice esattamente come richiesto
        content = strjoin({ ...
'File {', ...
' Grid = "n1_msh.tdr"', ...
' Plot = "n@node@_device_des.tdr"', ...
' Current = "n@node@_device_des.plt"', ...
' Parameter = "sdevice.par"', ...
'}', ...
'', ...
'Electrode {', ...
' { Name = "BottomContact" Voltage = 0 }', ...
' { Name = "TopContact" Voltage = 0 }', ...
'}', ...
'', ...
'# ----------------------------', ...
'# PHYSICS', ...
'# ----------------------------', ...
'# MODELS FOR MATERIALS', ...
'Physics (Material="HfO2") {', ...
' CondInsulator', ...
'}', ...
'', ...
'Physics (Material="SiO2") {', ...
' CondInsulator', ...
'}', ...
'', ...
'# THERMIC CONTACT', ...
'Thermode {', ...
' { Name = "BottomContact" Temperature = 300 SurfaceResistance = 1e-5 }', ...
' { Name = "TopContact" Temperature = 300 SurfaceResistance = 1e-5 }', ...
'}', ...
'', ...
'# ----------------------------', ...
'# PLOT', ...
'# ----------------------------', ...
'Plot {', ...
' Potential', ...
' ElectricField', ...
' DielectricConstant', ...
' Temperature', ...
'', ...
' # DIELECTRIC CURRENTS', ...
' ConductionCurrentDensity/Vector', ...
' DisplacementCurrentDensity/Vector', ...
'', ...
' # TOTAL CURRENT', ...
' TotalCurrent/Vector', ...
'', ...
' # OTHERS', ...
' SpaceCharge', ...
' Potential Doping', ...
' BandGap ElectronAffinity', ...
' ConductionBandEnergy ValenceBandEnergy', ...
'}', ...
'', ...
'Math {', ...
' RelErrControl', ...
' Extrapolate', ...
'}', ...
'', ...
'# ----------------------------', ...
'# SOLVE', ...
'# ----------------------------', ...
'Solve', ...
'{', ...
'Coupled (Iterations= 100 LineSearchDamping= 1e-8) {Poisson}', ...
'Coupled{ Poisson Electron Hole Temperature Contact CondInsulator }', ...
'Plot(FilePrefix="n@node@_equilibrium")', ...
'quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4', ...
'Goal {name= "TopContact" voltage = 10}', ...
'plot { range=(0, 1) intervals= 1}', ...
'', ...
'){coupled { Poisson Electron Hole Temperature CondInsulator }}', ...
'}' ...
        }, newline);
        [f,p] = uiputfile('Sdevice.txt','Export Sdevice.txt');
if isequal(f,0), return; end
        fid = fopen(fullfile(p,f),'w');
if fid<0
            uialert(ui,'Impossibile creare il file di output.','Errore','Icon','error');
return;
end
        fwrite(fid, content, 'char');
        fclose(fid);
        uialert(ui, sprintf('File generato:\n%s', fullfile(p,f)), 'Export completato', 'Icon', 'success');
end
end
% ====== Subfunction: disegno (preview) ======
function drawStack(ax, P)
% Colori e stile
faceColor_base = [0.55 0.35 0.20]; % SiO2
faceColor_Ti = [0.20 0.20 0.20]; % Ti
faceColor_Au = [1.00 0.84 0.00]; % Au
faceColor_vac = [0.00 0.60 0.00]; % vacuum
alpha_vac = 0.35;
faceColor_hfo2 = [0.80 0.65 0.45]; % HfO2
edgeColor = 'k';
lineWidth = 1.0;
vertices_box = @(o,Lx,Wy,Hz) [ o; ...
                               o + [Lx 0 0]; ...
                               o + [0 Wy 0]; ...
                               o + [0 0 Hz]; ...
                               o + [Lx Wy 0]; ...
                               o + [Lx 0 Hz]; ...
                               o + [0 Wy Hz]; ...
                               o + [Lx Wy Hz] ];
F = [1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
if ~isgraphics(ax), ax = gca; end
axes(ax);
delete(findall(ax,'type','legend'));
cla(ax,'reset');
ax.NextPlot = 'add';
% Geometria base
o = P.origin(:).';
V_base = vertices_box(o, P.L, P.W, P.H);
o_Ti = o + [0 0 P.H];
V_Ti = vertices_box(o_Ti, P.L, P.W, P.H_Ti_bottom);
o_Au = o + [0 0 (P.H + P.H_Ti_bottom)];
V_Au = vertices_box(o_Au, P.L, P.W, P.H_gold_bottom);
z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;
L_vac_eff = min(max(P.L_vac,0), P.L);
L_left = max(0, (P.L - L_vac_eff)/2);
L_right = max(0, P.L - (L_left + L_vac_eff));
o_left = [o(1), o(2), z0_top];
o_vac = [o(1) + L_left, o(2), z0_top];
o_right = [o(1) + L_left + L_vac_eff, o(2), z0_top];
V_left = vertices_box(o_left, L_left, P.W, P.H_vac);
V_vac = vertices_box(o_vac, L_vac_eff, P.W, P.H_vac);
V_right = vertices_box(o_right, L_right, P.W, P.H_vac);
% Au top centrato
x0_cap = o(1) + (P.L - P.L_Au_top)/2;
y0_cap = o(2) + (P.W - P.W_Au_top)/2;
z0_cap = z0_top + P.H_vac;
o_cap = [x0_cap, y0_cap, z0_cap];
V_cap = vertices_box(o_cap, P.L_Au_top, P.W_Au_top, P.H_Au_top);
% Pellicola Ti attorno all'Au top
o_film_xL = [x0_cap - P.T_film, y0_cap, z0_cap];
o_film_xR = [x0_cap + P.L_Au_top, y0_cap, z0_cap];
V_film_xL = vertices_box(o_film_xL, P.T_film, P.W_Au_top, P.H_Au_top);
V_film_xR = vertices_box(o_film_xR, P.T_film, P.W_Au_top, P.H_Au_top);
o_film_yF = [x0_cap - P.T_film, y0_cap - P.T_film, z0_cap];
o_film_yB = [x0_cap - P.T_film, y0_cap + P.W_Au_top, z0_cap];
V_film_yF = vertices_box(o_film_yF, P.L_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);
V_film_yB = vertices_box(o_film_yB, P.L_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);
% Cornice SiO2
x_in_min = max(x0_cap - P.T_film, o(1));
x_in_max = min(x0_cap + P.L_Au_top + P.T_film, o(1)+P.L);
y_in_min = max(y0_cap - P.T_film, o(2));
y_in_max = min(y0_cap + P.W_Au_top + P.T_film, o(2)+P.W);
T_left_frame = max(0, x_in_min - o(1));
T_right_frame = max(0, (o(1)+P.L) - x_in_max);
T_front_frame = max(0, y_in_min - o(2));
T_back_frame = max(0, (o(2)+P.W) - y_in_max);
z0_frame = z0_cap;
H_frame = P.H_Au_top;
V_frame_left = []; V_frame_right = []; V_frame_front = []; V_frame_back = [];
if T_left_frame > 0, V_frame_left = vertices_box([o(1), o(2), z0_frame], T_left_frame, P.W, H_frame); end
if T_right_frame > 0, V_frame_right = vertices_box([x_in_max, o(2), z0_frame], T_right_frame, P.W, H_frame); end
if T_front_frame > 0, V_frame_front = vertices_box([x_in_min, o(2), z0_frame], max(0, x_in_max - x_in_min), T_front_frame, H_frame); end
if T_back_frame > 0, V_frame_back = vertices_box([x_in_min, y_in_max, z0_frame], max(0, x_in_max - x_in_min), T_back_frame, H_frame); end
% Disegno
p1 = patch(ax,'Vertices',V_base,'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','SiO2');
p2 = patch(ax,'Vertices',V_Ti, 'Faces',F,'FaceColor',faceColor_Ti, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','Ti');
p3 = patch(ax,'Vertices',V_Au, 'Faces',F,'FaceColor',faceColor_Au, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','Au');
p4 = [];
if L_left > 0
    p4 = patch(ax,'Vertices',V_left,'Faces',F,'FaceColor',faceColor_hfo2,'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','HfO2');
end
p5 = [];
if L_vac_eff > 0
    p5 = patch(ax,'Vertices',V_vac,'Faces',F,'FaceColor',faceColor_vac,'FaceAlpha',alpha_vac,'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','vacuum');
end
if L_right > 0
    patch(ax,'Vertices',V_right,'Faces',F,'FaceColor',faceColor_hfo2,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
end
% Au top + pellicola Ti (fuori legenda)
patch(ax,'Vertices',V_cap,'Faces',F,'FaceColor',faceColor_Au,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
patch(ax,'Vertices',V_film_xL,'Faces',F,'FaceColor',faceColor_Ti,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
patch(ax,'Vertices',V_film_xR,'Faces',F,'FaceColor',faceColor_Ti,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
patch(ax,'Vertices',V_film_yF,'Faces',F,'FaceColor',faceColor_Ti,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
patch(ax,'Vertices',V_film_yB,'Faces',F,'FaceColor',faceColor_Ti,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
% Cornice SiO2 (fuori legenda)
if ~isempty(V_frame_left), patch(ax,'Vertices',V_frame_left, 'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off'); end
if ~isempty(V_frame_right), patch(ax,'Vertices',V_frame_right,'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off'); end
if ~isempty(V_frame_front), patch(ax,'Vertices',V_frame_front,'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off'); end
if ~isempty(V_frame_back), patch(ax,'Vertices',V_frame_back, 'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off'); end
axis(ax,'equal'); grid(ax,'on'); view(ax,35,25);
xlabel(ax,'X [nm]'); ylabel(ax,'Y [nm]'); zlabel(ax,'Z [nm]');
title(ax,'Standard Cell');
H_tot = P.H + P.H_Ti_bottom + P.H_gold_bottom + P.H_vac + P.H_Au_top;
marg = 0.1 * [P.L P.W H_tot];
xlim(ax,[o(1)-marg(1), o(1)+P.L+marg(1)]);
ylim(ax,[o(2)-marg(2), o(2)+P.W+marg(2)]);
zlim(ax,[o(3)-marg(3), o(3)+H_tot+marg(3)]);
plot3(ax,o(1),o(2),o(3),'ko','MarkerFaceColor','k','HandleVisibility','off');
quiver3(ax,o(1),o(2),o(3), P.L,0,0, 0, 'LineWidth',1, 'HandleVisibility','off');
quiver3(ax,o(1),o(2),o(3), 0,P.W,0, 0, 'LineWidth',1, 'HandleVisibility','off');
quiver3(ax,o(1),o(2),o(3), 0,0,H_tot, 0, 'LineWidth',1, 'HandleVisibility','off');
lh = gobjects(0);
if isgraphics(p1), lh(end+1) = p1; end
if isgraphics(p2), lh(end+1) = p2; end
if isgraphics(p3), lh(end+1) = p3; end
if isgraphics(p4), lh(end+1) = p4; end
if ~isempty(p5) && isgraphics(p5), lh(end+1) = p5; end
legend(ax, lh, 'Location','bestoutside');
end